#!/usr/bin/python

import read_amber as ra
import networkx as nx
import sys

class GhostAtom(ra.Atom):
  pass

valences = { "H": 1,
             "C": 4,
             "N": 3,
             "O": 2,
             "S": 2,
             "Se": 2,
             "P": 5 }

def peptide_bonds(atoms):
  peptide_bond_list = []
  amide_carbons = []
# Identify amide carbons.
  for carbon in [atom for atom in atoms if atom.element is "C"]:
    if isinstance(carbon, GhostAtom):
      continue
    if carbon.residue.name == 'PRO':
      continue
    neighbors = atoms.neighbors(carbon)
    nitrogens = [neighbor for neighbor in neighbors if neighbor.element is "N"]
    oxygens = [neighbor for neighbor in neighbors if neighbor.element is "O" and 
               not isinstance(neighbor, GhostAtom)]
    ghost_oxygens = [neighbor for neighbor in neighbors if neighbor.element is "O" and 
                     isinstance(neighbor, GhostAtom)]
    if len(nitrogens) == 1 and len(oxygens) == 1 and len(ghost_oxygens) == 1:
      amide_carbons.append(carbon)
# Identify the Ca - C - N - Ca chain.
  for carbon in amide_carbons:
    neighbors = atoms.neighbors(carbon)
    c_alpha_1 = [neighbor for neighbor in neighbors if neighbor.element is "C"][0]
    nitrogen = [neighbor for neighbor in neighbors if neighbor.element is "N"][0]
    n_neighbors = atoms.neighbors(nitrogen)
    c_alpha_2 = [neighbor for neighbor in n_neighbors if list(set(atoms.neighbors(neighbor)) & set(amide_carbons))]
    if len(c_alpha_2) == 1:
      peptide_bond_list.append((c_alpha_1, carbon, nitrogen, c_alpha_2[0]))
  return peptide_bond_list

def multi_bonds(atoms):
  multibonded = [atom for atom in atoms.nodes() 
                 if len(nx.edges(atoms, atom)) < valences[atom.element]]
  for i, atom in enumerate(multibonded):
    paired = False
    for other in atoms.neighbors(atom):
      if isinstance(other, GhostAtom):
        paired = True
        continue
      if len(nx.edges(atoms, other)) < valences[other.element]:
        ghost_atom = GhostAtom(**(atom.__dict__))
        ghost_atom.name = atom.name + "*"
        ghost_other = GhostAtom(**(other.__dict__))
        ghost_other.name = other.name + "*"
        atoms.add_edge(other, ghost_atom)
        atoms.add_edge(atom, ghost_other)
        paired = True
#    if not paired:
#      print "Could not find pair for atom", atom, atom.residue, atom.charge, nx.edges(atoms, atom)
  
def write_cis_trans_file(input_filename, output_filename):
  molecule = ra.parse_topology_file(input_filename)
  atoms = molecule.atoms
#  for atom in atoms:
#    print atom.residue
  multi_bonds(atoms)
  with open(output_filename, "w") as output_file:
    for bond in sorted(peptide_bonds(atoms), cmp=lambda x, y: cmp(x[0].index, y[0].index)):
    # Write out the list of atoms in peptide bonds (Ca - C - N - Ca).
      output_string = "{0:>8d}{1:>8d}{2:>8d}{3:>8d}\n".format(*map(lambda x: x.index + 1, bond))
      output_file.write(output_string)

if __name__ == "__main__":
  write_cis_trans_file(sys.argv[1], ".cis_trans_list")
