import argparse

import networkx as nx

from pele.storage.database import Database
from pele.utils.disconnectivity_graph import database2graph

def print_system_properties(db, supress_long=True):
    if len(db.properties()) == 0: return
    print "System properties:"
    print "------------------"
    for p in db.properties():
        name, value = p.name(), p.value()
        str_value = str(value)
        if len(str_value) > 100 and supress_long:
            str_value = str_value[:80] + " '... output suppressed'"
        print "%10s:\t\t%s" % (name, str_value)
    print ""
        

def long_summary(db):
    nts = db.number_of_transition_states()
    if nts == 0:
        print "long summary not applicable: no transition states"
        return
    graph = database2graph(db)
    
    cclist = nx.connected_components(graph)
    
#    print "number of connected components", len(cclist)
    counts = dict()
    minimum_energy = dict()
    for cc in cclist:
        nc = len(cc)
        Emin = min((m.energy for m in cc))
        try:
            counts[nc] += 1
            if Emin < minimum_energy[nc]:
                minimum_energy[nc] = Emin 
        except KeyError:
            counts[nc] = 1
            minimum_energy[nc] = Emin
    counts = counts.items()
    counts.sort(key=lambda x:-x[0])
    print "Connectivity of the database:"
    for n, count in counts:
        if n == 1:
            print "%7d unconnected minima                : minimum energy = %s" % (count, minimum_energy[n])
        else:
            print "%7d connected clusters of size %7d: minimum energy = %s" % (count, n, minimum_energy[n]) 


def main():
    parser = argparse.ArgumentParser(description="print information about the database")

    parser.add_argument("database", type=str, help="Database file name")
    
    parser.add_argument("--write-disconnect",
                      dest="writeDPS", action="store_true",
                      help="generate min.dat and ts.dat to use with disconnectDPS")
    parser.add_argument("-m",
                      dest="writeMinima", action="store_true",
                      help="dump minima to screen")
    parser.add_argument("-t",
                      dest="writeTS", action="store_true",
                      help="dump transition states to screen")
    parser.add_argument("-p",
                      dest="properties", action="store_true",
                      help="print system properties")
    parser.add_argument("-s",
                      dest="summary", action="store_true",
                      help="print summary")
    parser.add_argument("-S",
                      dest="summary_long", action="store_true",
                      help="print long summary")
    args = parser.parse_args()
    
    if args.summary_long:
        args.summary = True
        
    db = Database(db=args.database, createdb=False)

    if args.properties or args.summary:
        print_system_properties(db)

    if args.summary:
        print "number of minima:", db.number_of_minima()
        print "number of transition states:", db.number_of_transition_states()
  
    if args.summary_long:
        long_summary(db)
        
    if args.writeMinima:
        print "List of minima: energy id fvib pgorder"
        print "---------------"
        for m in db.minima():
            print "%f\t\tid %d %s %s" % (m.energy, m._id, str(m.fvib), str(m.pgorder))
        print "END\n"
    
    if args.writeTS:
        print "List of transition states:"
        print "--------------------------"
        for ts in db.transition_states():
            print "%d\t<->\t%d\tid %d\tenergies %f %f %f" % \
                (ts.minimum1._id, ts.minimum2._id, ts._id, ts.minimum1.energy, ts.energy, ts.minimum2.energy)
        print "END\n"

    if args.writeDPS:
        writeDPS(db)
        

def writeDPS(db):
    minindex={}
    out = open("min.data", "w")
    i=1
    for m in db.minima():
        minindex[m]=i
        i+=1
        if m.fvib is None:
            print "no frequency information available for minimum", m._id
        if m.pgorder is None:
            print "no pgorder information available for minimum", m._id
        out.write("%f %f %d 0.0 0.0 0.0\n"%(m.energy, m.fvib, m.pgorder))
    out = open("ts.data", "w")
    ti=0
    for ts in db.transition_states():
        ti+=1
        out.write("%f %f %d %d %d 0.0 0.0 0.0\n"%(ts.energy, ts.fvib, ts.pgorder, minindex[ts.minimum1], minindex[ts.minimum2]))
    print "Written %d minima and %d transition states"%(i, ti)

if __name__ == "__main__":
    main()    
